from pandac.PandaModules import Vec3, NodePath
from GridChild import GridChild

class CartesianGridBase:
    __module__ = __name__

    def __init__(self, startingZone=500, gridSize=100, gridRadius=3, cellWidth=50, style='Cartesian', period=0):
        self.startingZone = startingZone
        self.gridSize = gridSize
        self.gridRadius = gridRadius
        self.cellWidth = cellWidth
        self.style = style
        self.center = gridSize * cellWidth / 2.0
        self.__managedChildren = set()
        self.__managementTask = None
        self.__period = period
        return

    def disable(self):
        self.stopManagementTask()
        self.__managedChildren.clear()

    def isGrid(self):
        return True

    def setParentingRules(self, style, rule):
        rules = rule.split(':')
        self.style = style
        self.startingZone = int(rules[0])
        self.gridSize = int(rules[1])
        self.viewingRadius = int(rules[2])
        self.center = self.gridSize * self.cellWidth / 2.0

    def getParentingRules(self):
        rule = '%i:%i:%i' % (self.startingZone, self.gridSize, self.gridRadius)
        return [self.style, rule]

    def getCellWidth(self):
        return self.cellWidth

    def setCellWidth(self, width):
        self.cellWidth = width

    def isValidZone(self, zoneId):
        if self.style == 'Cartesian':
            return self.isGridZone(zoneId)
        elif self.style == 'CartesianStated':
            return 0 <= zoneId < self.startingZone or self.isGridZone(zoneId)
        else:
            return False

    def isGridZone(self, zoneId):
        return self.startingZone <= zoneId < self.startingZone + self.gridSize * self.gridSize

    def getZoneFromXYZ(self, pos, wantRowAndCol=False):
        x = pos[0] + self.center
        y = pos[1] + self.center
        col = x // self.cellWidth
        row = y // self.cellWidth
        zoneId = int(self.startingZone + (row * self.gridSize + col))
        if wantRowAndCol:
            return (
             zoneId, col, row)
        else:
            return zoneId

    def getZoneAndCellPosFromXYZ(self, pos):
        zone = self.getZoneFromXYZ(pos)
        x = pos[0] % self.cellWidth
        y = pos[1] % self.cellWidth
        return (zone, Vec3(x, y, pos[2]))

    def getGridSizeFromSphereRadius(self, sphereRadius, cellWidth, gridRadius):
        sphereRadius = max(sphereRadius, gridRadius * cellWidth)
        return 2 * (sphereRadius // cellWidth)

    def getGridSizeFromSphere(self, sphereRadius, spherePos, cellWidth, gridRadius):
        xMax = abs(spherePos[0]) + sphereRadius
        yMax = abs(spherePos[1]) + sphereRadius
        sphereRadius = Vec3(xMax, yMax, 0).length()
        return max(2 * (sphereRadius // cellWidth), 1)

    def getZoneCellOrigin(self, zoneId):
        zone = zoneId - self.startingZone
        row = zone // self.gridSize
        col = zone % self.gridSize
        x = col * self.cellWidth - self.center
        y = row * self.cellWidth - self.center
        return (
         x, y, 0)

    def getZoneCellOriginCenter(self, zoneId):
        (x, y, z) = self.getZoneCellOrigin(zoneId)
        center = self.cellWidth * 0.5
        return (x + center, y + center, z)

    def getConcentricZones(self, zoneId, radius):
        zones = []
        zone = zoneId - self.startingZone
        row = zone // self.gridSize
        col = zone % self.gridSize
        leftOffset = min(col, radius)
        rightOffset = min(self.gridSize - (col + 1), radius)
        topOffset = min(row, radius)
        bottomOffset = min(self.gridSize - (row + 1), radius)
        ulZone = zoneId - leftOffset - topOffset * self.gridSize
        for currCol in range(int(rightOffset + leftOffset + 1)):
            if currCol == 0 and leftOffset == radius or currCol == rightOffset + leftOffset and rightOffset == radius:
                possibleRows = range(int(bottomOffset + topOffset + 1))
            else:
                possibleRows = []
                if topOffset == radius:
                    possibleRows.append(0)
                if bottomOffset == radius:
                    possibleRows.append(bottomOffset + topOffset)
            for currRow in possibleRows:
                newZone = ulZone + currRow * self.gridSize + currCol
                zones.append(int(newZone))

        return zones

    def setChildGridCells(self, child, zoneId):
        child.setGridCell(self, zoneId)

    def parentObjectToArea(self, child):
        child.reparentTo(self)
        zoneId = self.getZoneFromXYZ(child.getPos())
        if self.isGridZone(zoneId):
            self.setChildGridCells(child, zoneId)
            return zoneId
        return 0

    def handleChildArrive(self, child, zoneId):
        if self.isGridZone(zoneId):
            child.setGridCell(self, zoneId)
        else:
            child.setGridCell(None, 0)
        return

    def handleChildLeave(self, child, zoneId):
        self.ignoreChild(child)
        child.setGridCell(None, 0)
        return

    def manageChild(self, child):
        self.__managedChildren.add(child)
        self.__manageChild(child, setup=True)
        if not self.__managementTask:
            self.startManagementTask()

    def ignoreChild(self, child):
        self.__managedChildren.discard(child)
        if not self.__managedChildren:
            self.stopManagementTask()

    def startManagementTask(self):
        self.stopManagementTask()
        if self.__period:
            self.__managementTask = taskMgr.doMethodLater(self.__period, self.__manage, self.taskName('ManageGrid'))
            self.__managementTask._return = self.__managementTask.again
        else:
            self.__managementTask = taskMgr.add(self.__manage, self.taskName('ManageGrid'))
            self.__managementTask._return = self.__managementTask.cont

    def __manage(self, task):
        for child in self.__managedChildren:
            self.__manageChild(child)

        return task._return

    def validGridInterest(self, child, gridId):
        return True

    def __manageChild(self, child, setup=False):
        gridIds = child.getGridInterestIds()
        for currGridId in gridIds:
            currGrid = getBase().getRepository().doId2do.get(currGridId)
            if not currGrid:
                continue
            adjustGrid = False
            newZoneId = None
            if currGrid is self:
                (x, y, z) = child.getPos()
                if x < 0 or y < 0 or x > currGrid.cellWidth or y > currGrid.cellWidth or setup:
                    adjustGrid = True
            else:
                newZoneId = currGrid.getZoneFromXYZ(child.getPos(currGrid))
                if newZoneId != child.getGridInterestZoneId(currGridId):
                    adjustGrid = True
            if adjustGrid:
                if not newZoneId:
                    newZoneId = currGrid.getZoneFromXYZ(child.getPos(currGrid))
                if currGrid.isGridZone(newZoneId):
                    if currGrid is self:
                        child.setGridCell(self, newZoneId, updateInterest=True)
                        child.checkPosition()
                        child.sendCurrentPosition()
                        if not self.validGridInterest(child, currGridId):
                            self.notify.warning('skipping setLocation with child %s and grid %s' % (child, currGridId))
                            continue
                        if config.GetBool('print-interest-debug', 1) and game.process == 'client' and launcher.getValue('GAME_ENVIRONMENT', 'DEV') in ['QA', 'DEV']:
                            base.cr.printInterestSets()
                            self.notify.warning('pre-setLocation: interests from child %s are %s' % (child.doId, child._gridInterests))
                        child.b_setLocation(self.getDoId(), newZoneId)
                    else:
                        child.updateGridInterest(currGrid, newZoneId)
                else:
                    self.notify.warning('%s handleChildCellChange %s: not a valid zone (%s) for pos %s' % (self.doId, child.doId, newZoneId, child.getPos(currGrid)))

        return

    def stopManagementTask(self):
        if self.__managementTask:
            taskMgr.remove(self.__managementTask)
        self.__managementTask = None
        return

    def taskName(self, taskString):
        pass

    def forceManagePass(self, child, setup=False):
        if child in self.__managedChildren:
            self.__manageChild(child, setup=setup)